classdef class_REVS_CVM_result < class_REVS_VM_result
    
	properties
		% Required Parameters	
		map_fuel;								% Test Fuel Properties
		output_fuel;							% Fuel Properties to use for reporting - adjust for heating value
		
		% Base Properties - set by model	
		phase_engine_revolutions;
		total_engine_revolutions;
		
		phase_engine_running_time_secs;
		total_engine_running_time_secs;
		
		phase_crankshaft_gross_pos_work_kJ;
		total_crankshaft_gross_pos_work_kJ;
		
		phase_crankshaft_output_pos_work_kJ;
		total_crankshaft_output_pos_work_kJ;
		
		phase_transmission_output_pos_work_kJ;
		total_transmission_output_pos_work_kJ;
		
		phase_axle_output_pos_work_kJ;
		total_axle_output_pos_work_kJ;
		
		phase_battery_energy_delta_kJ;			% Phase Battery Stored Energy Delta [kJ]
		total_battery_energy_delta_kJ;			% Total Battery Stored Energy Delta [kJ]
		
        drive_quality;                          % structure of SAEJ2951 drive quality metrics, from run_cycle_postproc
	end
	
	properties (Hidden = true )
		
		raw_phase_fuel_consumed_g;				% Where actual model result gets stored
		raw_total_fuel_consumed_g;				% Where actual model result gets stored
		
	end
	
	properties ( Dependent= true , Transient = true)
			
		phase_crankshaft_gross_pos_work_kWh;
		total_crankshaft_gross_pos_work_kWh;
		
		phase_crankshaft_output_pos_work_kWh;
		total_crankshaft_output_pos_work_kWh;
		
		phase_transmission_output_pos_work_kWh;
		total_transmission_output_pos_work_kWh;
		
		phase_axle_output_pos_work_kWh;
		total_axle_output_pos_work_kWh;
		
		phase_fuel_consumed_g;
		total_fuel_consumed_g;
		
		phase_fuel_consumed_vol_gallons;
		total_fuel_consumed_vol_gallons;

        phase_fuel_consumed_CFR_gallons;
		total_fuel_consumed_CFR_gallons;
		
		phase_gCO2;
		total_gCO2;
		
		phase_gCO2pmi;
		total_gCO2pmi;
		
		phase_FE_vol_mpg;
		total_FE_vol_mpg;
		
		phase_FE_CFR_mpg;
		total_FE_CFR_mpg;
		
	
		phase_engine_off_time_secs;
		total_engine_off_time_secs;
		
		phase_engine_off_time_pct;
		total_engine_off_time_pct;
		
		phase_engine_running_time_pct;
		total_engine_running_time_pct;
				
	end
	
	
	methods
		
		%% Constructor
		function obj = class_REVS_CVM_result( drive_cycle, engine_fuel, vehicle_fuel)
			
			
			if isfield( drive_cycle,'name')
				obj.drive_cycle_name = drive_cycle.name;
			end
			
			if isfield( drive_cycle,'phase_name')
				obj.drive_cycle_phase_names = drive_cycle.phase_name;
			end
			
			obj.map_fuel = engine_fuel;
			obj.output_fuel = engine_fuel;
			
			if ~isempty( vehicle_fuel )
				obj.output_fuel = vehicle_fuel;
			end
			
		end
		
		%% Print Result - to screen or file
		function print(obj, fid )
			
			if nargin < 2 || isempty(fid)
				fid = 1;
			end
			
			% ... display percent time trace missed by 2mph (%)
			fprintf(fid,'\n-- Summary data for %s drive cycle --\n\n',    obj.drive_cycle_name);
			fprintf(fid,'   Percent Time Missed by 2mph   = %6.2f %%\n',  obj.total_driver_error_time_secs ./ obj.total_time_secs);
			
			
			if length(obj.phase_time_secs) ~= 1 ||  obj.phase_time_secs ~= obj.total_time_secs
				for p = 1:length(obj.phase_time_secs)
					
					if ~isempty( obj.drive_cycle_phase_names ) && p <= length(obj.drive_cycle_phase_names )
						phase_name = obj.drive_cycle_phase_names{p};
					else
						phase_name = '';
					end
					
					fprintf(fid,'   Phase %2d: %s %s\n',p, phase_name, repmat('-', 1,25-length(phase_name)));
					fprintf(fid,'   Distance                    = %6.3f mi\n',			obj.phase_distance_mi(p));
					fprintf(fid,'   Fuel Consumption            = %6.4f grams\n',		obj.phase_fuel_consumed_g(p));
					fprintf(fid,'   Fuel Consumption            = %6.4f gallons\n',		obj.phase_fuel_consumed_vol_gallons(p));
					fprintf(fid,'   Fuel Economy (Volumetric)   = %6.3f mpg\n',			obj.phase_FE_vol_mpg(p));
					fprintf(fid,'   Fuel Economy (CFR)          = %6.3f mpg\n',			obj.phase_FE_CFR_mpg(p));
					fprintf(fid,'   Fuel Consumption            = %6.3f g/mile\n',		obj.phase_fuel_consumed_g(p) / obj.phase_distance_mi(p));
					fprintf(fid,'   CO2 Emission                = %6.2f g/mile\n',		obj.phase_gCO2pmi(p));
					fprintf(fid,'\n');
					
				end
			end
			
			% Print Total Results
			fprintf(fid,'   Total ---------------------------------------\n');
			fprintf(fid,'   >> %s\n', obj.drive_cycle_name);
			fprintf(fid,'   Distance                    = %6.3f mi\n',		obj.total_distance_mi);
			fprintf(fid,'   Fuel Consumption            = %6.4f grams\n',	obj.total_fuel_consumed_g);
			fprintf(fid,'   Fuel Consumption (Vol.)     = %6.4f gallons\n', obj.total_fuel_consumed_vol_gallons);
			fprintf(fid,'   Fuel Economy (Vol.)         = %6.3f mpg\n',		obj.total_FE_vol_mpg);            
			fprintf(fid,'   Fuel Consumption (CFR)      = %6.4f gallons\n', obj.total_fuel_consumed_CFR_gallons);
			fprintf(fid,'   Fuel Economy (CFR)          = %6.3f mpg\n',		obj.total_FE_CFR_mpg);
			fprintf(fid,'   Fuel Consumption            = %6.3f g/mile\n', obj.total_fuel_consumed_g / obj.total_distance_mi);
			fprintf(fid,'   CO2 Emission                = %6.2f g/mile\n', obj.total_gCO2pmi);
			fprintf(fid,'\n');
		end
		
		
		%% Fuel
		function obj = set.phase_fuel_consumed_g(obj, val)
			obj.raw_phase_fuel_consumed_g = val;
		end
		
		function obj = set.total_fuel_consumed_g(obj, val)
			obj.raw_total_fuel_consumed_g = val;
		end
		
		function val = get.phase_fuel_consumed_g(obj)
			val = obj.raw_phase_fuel_consumed_g .* obj.map_fuel.energy_density_MJpkg ./ obj.output_fuel.energy_density_MJpkg; % Conversion @ R = 1
		end
		
		function val = get.total_fuel_consumed_g(obj)
			val = obj.raw_total_fuel_consumed_g.* obj.map_fuel.energy_density_MJpkg ./ obj.output_fuel.energy_density_MJpkg; % Conversion @ R = 1
		end
		
		function val = get.phase_fuel_consumed_vol_gallons(obj)
			val = obj.phase_fuel_consumed_g ./ 1000 ./ obj.output_fuel.density_kgpL_15C .* convert.lit2gal ;
		end
		
		function val = get.total_fuel_consumed_vol_gallons(obj)
			val = obj.total_fuel_consumed_g ./ 1000 ./ obj.output_fuel.density_kgpL_15C .* convert.lit2gal;
		end
		
		function val = get.phase_fuel_consumed_CFR_gallons(obj)
			val = obj.phase_distance_mi ./ obj.phase_FE_CFR_mpg;
		end
		
		function val = get.total_fuel_consumed_CFR_gallons(obj)
			val = obj.total_distance_mi ./ obj.total_FE_CFR_mpg;
		end
		
		
		%% Fuel Economy
		function val = get.phase_FE_vol_mpg(obj)
			val = obj.phase_distance_mi ./ obj.phase_fuel_consumed_vol_gallons;
        end
		
		function val = get.total_FE_vol_mpg(obj)
			val = obj.total_distance_mi ./ obj.total_fuel_consumed_vol_gallons;
		end
		
		function val = get.phase_FE_CFR_mpg(obj)
			val = obj.phase_distance_mi .* 5174e4 .* obj.output_fuel.specific_gravity  ./ ( obj.phase_fuel_consumed_g .* ( 0.6 * obj.output_fuel.specific_gravity * obj.output_fuel.energy_density_MJpkg * convert.MJpkg2BTUplbm + 5471));
        end
		
		function val = get.total_FE_CFR_mpg(obj)
			val = obj.total_distance_mi .* 5174e4 .* obj.output_fuel.specific_gravity  ./ ( obj.total_fuel_consumed_g .* ( 0.6 * obj.output_fuel.specific_gravity * obj.output_fuel.energy_density_MJpkg * convert.MJpkg2BTUplbm + 5471));
		end
		
		%% CO2
		function val = get.phase_gCO2(obj)
			val = obj.phase_fuel_consumed_g .* obj.output_fuel.carbon_weight_fraction * (44.0095/12.0107);
		end
		
		function val = get.total_gCO2(obj)
			val = obj.total_fuel_consumed_g .* obj.output_fuel.carbon_weight_fraction * (44.0095/12.0107);
		end
		
		function val = get.phase_gCO2pmi(obj)
			val = obj.phase_gCO2 ./ obj.phase_distance_mi ;
		end
		
		function val = get.total_gCO2pmi(obj)
			val = obj.total_gCO2 ./ obj.total_distance_mi ;
		end
		
				
		%% Work
		function val = get.phase_crankshaft_gross_pos_work_kWh(obj)
			val = obj.phase_crankshaft_gross_pos_work_kJ / 3600;
		end
		
		function val = get.total_crankshaft_gross_pos_work_kWh(obj)
			val = obj.total_crankshaft_gross_pos_work_kJ / 3600;
		end
		
		function val = get.phase_crankshaft_output_pos_work_kWh(obj)
			val = obj.phase_crankshaft_output_pos_work_kJ / 3600;
		end
		
		function val = get.total_crankshaft_output_pos_work_kWh(obj)
			val = obj.total_crankshaft_output_pos_work_kJ / 3600;
		end
		
		function val = get.phase_transmission_output_pos_work_kWh(obj)
			val = obj.phase_transmission_output_pos_work_kJ / 3600;
		end
		
		function val = get.total_transmission_output_pos_work_kWh(obj)
			val = obj.total_transmission_output_pos_work_kJ / 3600;
		end
		
		function val = get.phase_axle_output_pos_work_kWh(obj)
			val = obj.phase_axle_output_pos_work_kJ / 3600;
		end
		
		function val = get.total_axle_output_pos_work_kWh(obj)
			val = obj.total_axle_output_pos_work_kJ / 3600;
		end
		
		%% Engine Off
		function val = get.phase_engine_off_time_secs(obj)
			val = obj.phase_time_secs - obj.phase_engine_running_time_secs;
		end
		
		function val = get.total_engine_off_time_secs(obj)
			val = obj.total_time_secs - obj.total_engine_running_time_secs;
		end
		
		function val = get.phase_engine_off_time_pct(obj)
			val = obj.phase_engine_off_time_secs ./ obj.phase_time_secs;
		end
		
		function val = get.total_engine_off_time_pct(obj)
			val = obj.total_engine_off_time_secs ./ obj.total_time_secs;
		end
		
		function val = get.phase_engine_running_time_pct(obj)
			val = obj.phase_engine_running_time_secs ./ obj.phase_time_secs;
		end
		
		function val = get.total_engine_running_time_pct(obj)
			val = obj.total_engine_running_time_secs ./ obj.total_time_secs;
		end
		
	end
	
end

